// File: Grapher.cp
//
// Description: This file contains the implementation of class
//              'TGraph'. An instance of this class can be used
//              to graphically depict tree and DAG data structures.
//
// Copyright 1992, 1994,  Mark Watson
//

#include "Grapher.h"
#include <String.h>

#define X_OFF 12
#define Y_OFF 3


TGraph::TGraph(TAppWindow * myWind, char * root_name)
{
    if (myWind == (TAppWindow *)NULL) {
        Warning("Illegal TGraph::TGraph constructor call");
        exit(1);
    }
    myWindow = myWind;
    init_tree(root_name);
    char buf[256];
    sprintf(buf,"Data for root %s", root_name);
    t.nodes[0].private_data = new char[strlen(buf)+1];
    sprintf(t.nodes[0].private_data,"Data for root %s", root_name);

    for (int n=1; n<MAX_NODES; n++)
        t.nodes[n].private_data = (char *)NULL;

    // layout the nodes:
    do_layout();
}

TGraph::~TGraph()
{
    for (int i=0; i<t.num; i++)  delete t.nodes[i].private_data;
}

void TGraph::do_layout()
{
    t.parent_node = 0;
    for (int i=0; i<t.num; i++) {
        t.nodes[i].x = 0;
        t.nodes[i].y = 0;
        t.nodes[i].plot_node = 0;
    }
    last_y = 0;
    Y_layout(0, 0);
    X_layout(0, 0);
    for (i=0; i<t.num; i++) {
        t.nodes[i].selectionFlag = 0;
        t.nodes[i].x += X_OFF;
        t.nodes[i].y += Y_OFF;
    }
    Draw();
}

void TGraph::do_layout(int new_parent_node)
{
    t.parent_node = new_parent_node;
    for (int i=0; i<t.num; i++) {
        t.nodes[i].x = 0;
        t.nodes[i].y = 0;
        t.nodes[i].plot_node = 0;
    }
    last_y = 0;
    Y_layout(t.parent_node, 0);
    X_layout(t.parent_node, 0);
    for (i=0; i<t.num; i++) {
        t.nodes[i].selectionFlag = 0;
        t.nodes[i].x += X_OFF;
        t.nodes[i].y += Y_OFF;
    }
    Draw();
}

#define Y_SPACING 20

void TGraph::Draw()
{   int i, parent;
    char buf[80];

    myWindow->erase_rect(0,1024,1024,0);

    for (i=0; i<t.num; i++)  {
        if (t.nodes[i].parent_id > -1)  { /* plot a line between this node and parent */
            parent = t.nodes[i].parent_id;
            if (t.nodes[parent].plot_node == 1)
                myWindow->plot_line(t.nodes[parent].x, t.nodes[parent].y - (TEXT_SIZE / 2)+3,
                                    t.nodes[i].x, t.nodes[i].y - (TEXT_SIZE / 2)+3);
        }
    }
    for (i=0; i<t.num; i++)  {
        if (t.nodes[i].plot_node == 1) {
            if (t.nodes[i].selectionFlag == 1)
                sprintf(buf,"<%s>",t.nodes[i].name);
            else if (t.nodes[i].selectionFlag == 2)
                sprintf(buf,"** %s **",t.nodes[i].name);
            else if (t.nodes[i].selectionFlag == 3)
                sprintf(buf,"[%s]",t.nodes[i].name);
            else
                sprintf(buf,"%s",t.nodes[i].name);
            myWindow->plot_string(t.nodes[i].x, t.nodes[i].y - 4, buf);
        }
    }
}

void TGraph::init_tree(char *root_name)
{   int i;
    for (i=0; i<MAX_NODES; i++) {
        t.nodes[i].name[0] = '\0';
        t.nodes[i].private_data = (char *)NULL;
    }
    sprintf(t.nodes[0].name,"%s",root_name);
    t.num = 1;
    t.nodes[0].parent_id = -1;  /* special for root node */
    t.nodes[0].x = t.nodes[0].y =  0;
}

void TGraph::add_child(char *name, char *parent)
{   int parent_index, i;
    if (t.num < (MAX_NODES - 1)) {
        parent_index = -1000;
        for (i=0; i<t.num; i++)
            if (strcmp((char *)parent,(char *)t.nodes[i].name) == 0) {
                t.nodes[t.num].parent_id = i;
                sprintf(t.nodes[t.num].name,"%s",name);
                t.nodes[t.num].x = t.nodes[t.num].y = 0;
                t.num++;
                break;
            }
    }
}

int TGraph::name_to_id(char *name)
{   int i;
    for (i=0; i<t.num; i++)
        if (strcmp((char *)name, (char *)t.nodes[i].name) == 0)
            return i;
    return -1;
}

int TGraph::get_children(char *name, int *child_list, int buf_size)
{   int i, count, node_id;
    count = 0;
    node_id = name_to_id(name);
    if (node_id > -1) {
        for (i=0; i<t.num; i++)
            if (node_id == t.nodes[i].parent_id)
                if (count < (buf_size - 1))
                    child_list[count++] = i;
    }
    return count;
}

void TGraph::Y_layout(int node_id, int level)
{   int i, nc, clist[50], average_y;
    t.nodes[node_id].plot_node = 1;
    if (t.nodes[node_id].y == 0) {  /* not yet laid out */
        nc = get_children(t.nodes[node_id].name, clist, 50);
        if (nc > 0) { /* unlaid out children to process */
            for (i=0; i<nc; i++)
                Y_layout(clist[i], level+1);
            average_y = 0;
            for (i=0; i<nc; i++) average_y += t.nodes[clist[i]].y;
            average_y = average_y / nc;
            t.nodes[node_id].y = average_y;
        } else {  /* no children */
            t.nodes[node_id].y = last_y + Y_SPACING;
            last_y = t.nodes[node_id].y;
        }
            t.nodes[node_id].x = 0;
    }
}

void TGraph::X_layout(int node_id, int level)
{   int clist[50];
    if (level == 0) {
        t.nodes[node_id].x = 12;
        int nc = get_children(t.nodes[node_id].name, clist, 50);
        if (nc > 0) { /* unlaid out children to process */
            for (int i=0; i<nc; i++)
                X_layout(clist[i], level+1);
        }
    } else {
        // calculate x spacing based on length of parent node's name:
        int len = strlen(t.nodes[t.nodes[node_id].parent_id].name);
        t.nodes[node_id].x = t.nodes[t.nodes[node_id].parent_id].x  +
            12 + myWindow->string_width("A") * len;
        int nc = get_children(t.nodes[node_id].name, clist, 50);
        if (nc > 0) { /* unlaid out children to process */
            for (int i=0; i<nc; i++)
                X_layout(clist[i], level+1);
        }
    }
}


void TGraph::set_selection_children(int node_id)
{   int i, nc, clist[20];
    t.nodes[node_id].selectionFlag = 1;
    // set selection flag child nodes = 1:
    nc = get_children(t.nodes[node_id].name, clist, 20);
    for (i=0; i<nc; i++)
        set_selection_children(clist[i]);
}

void TGraph::set_selection_parent(int node_id)
{
    t.nodes[node_id].selectionFlag = 3;
    if (t.nodes[node_id].parent_id >= 0)
        set_selection_parent(t.nodes[node_id].parent_id);
}

void TGraph::setSelection(int node_id)
{   int i, nc;  int clist[20];
    // clear all selection flags:
    for (i=0; i<t.num; i++) t.nodes[i].selectionFlag = 0;
    if (node_id < 0)  return;
    // set selection flag child nodes = 1:
    nc = get_children(t.nodes[node_id].name, clist, 20);
    for (i=0; i<nc; i++)
        set_selection_children(clist[i]);
    // set selection flag of parent nodes = 3:
    if (t.nodes[node_id].parent_id >= 0)
        set_selection_parent(t.nodes[node_id].parent_id);
    // set this node's selection flag = 2:
    t.nodes[node_id].selectionFlag = 2;
}
	
int TGraph::getSelectedNode()
{
    for (int i=0; i<t.num; i++)
        if (t.nodes[i].selectionFlag == 2)  return i;
    return -1;
}

int TGraph::closestNode(int x, int y)
{
    long distX, distY;
    for (int i=0; i<t.num; i++)  {
        distX = (x  - 15) - (t.nodes[i].x - X_OFF); // offset for clicking node-center
        distY = y - (t.nodes[i].y - Y_OFF);
        distX = distX * distX / 3; // allow a larger error in x-direction
        distY = distY * distY;
        if ((distX + distY) < 60)  return i;
    }
    return -1;
}

void TGraph::set_private_data(int node_index, char *data)
{
    t.nodes[node_index].private_data = data;
}

char * TGraph::get_private_data(int node_index)
{
	if (node_index < t.num)
	   return t.nodes[node_index].private_data;
	else
	   return "";
}
